/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.Text;
using System.Drawing;
using System.Web;
using System.Web.UI;
using System.Web.UI.WebControls;
using System.Web.SessionState;
using Borland.Eco.UmlRt;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.Handles;
using System.Globalization;

namespace Borland.Eco.Web
{
	/// <summary>
	/// Summary description for MethodCallBuilder.
	/// </summary>
	public class MethodCallBuilder
	{
#region private helper classes
		private class ButtonExecuteMethod
		{
			private readonly IMethod m_Method;
			private readonly IExternalVariableList m_Variables;
			private readonly ElementHandle m_Handle;
			private readonly AutoForm m_AutoForm;
			private readonly IList parameterBindings;
			private readonly HttpResponse response;
			private readonly HttpRequest request;

			public  ButtonExecuteMethod(ElementHandle handle,
				IMethod method,
				IExternalVariableList variables,
				AutoForm autoForm,
				IList parameterBindings,
				HttpResponse response,
				HttpRequest request)
			{
				m_Handle = handle;
				m_Method = method;
				m_Variables = variables;
				m_AutoForm = autoForm;
				this.parameterBindings = parameterBindings;
				this.response = response;
				this.request = request;
			}

			public void OnClick(object sender, EventArgs args)
			{
				try
				{
					foreach (ParameterBinding pb in parameterBindings)
						pb.Apply() ;
					IElement[] parameters = new IElement[m_Variables.Count];
					for (int i=0; i<m_Variables.Count; i++)
						parameters[i] = m_Variables[i].Element;
					(m_Handle.Element as IObject).ObjectInstance.Invoke(m_Method, parameters);

					if (m_Method.EcoMethod.IsTrigger)
					{
						response.Redirect(request.Url.PathAndQuery);
						response.Flush();
						response.End();
					}
					else
					{
						m_AutoForm.SetMethodButtonsEnabledStatus();
						m_AutoForm.RefreshPropertyControls();
					}
				}
				catch (Exception e)
				{
					m_AutoForm.ShowMessage(WebStringRes.sOperationCallFailed + ": " + e.ToString()); // do not localize
				}
			}
		}

		private class ButtonExecuteAction
		{
			private readonly string m_Action;
			private readonly IExternalVariableList m_Variables;
			private readonly ElementHandle m_Handle;

			public  ButtonExecuteAction(ElementHandle handle,
				string action,
				IExternalVariableList variables)
			{
				m_Handle = handle;
				m_Action = action;
				m_Variables = variables;
			}

			public void OnClick(object sender, EventArgs args)
			{
				try
				{
					IActionLanguageService als = (m_Handle as IEcoServiceProvider).GetEcoService(typeof(IActionLanguageService)) as IActionLanguageService;
					als.Execute(m_Handle.Element, m_Action, m_Variables);
				}
				catch
				{
					;
				}
			}

		}


#endregion
		private const bool ShowParameterTypes = true;
		private const bool AutoScrollParameters = false;

		public static void BuildControls(Control hostControl, EcoSpace ecoSpace,
			ElementHandle handle, IClass umlClass, Table table, IList myMethodBindings, AutoForm autoForm, HttpResponse response, HttpRequest request, HttpSessionState session)
		{
			BuildControls(hostControl, ecoSpace, handle, umlClass, table, myMethodBindings, autoForm, "", response, request, session);
		}

		///<summary>
		///Builds controls for the methods of <paramref name="modelElement"/> on <paramref name="hostControl"/> and binds them to the <paramref name="handle"/>.
		///</summary>
		///<param name="hostControl">The control used as container for created controls.</param>
		///<param name="ecoSpace">AutoContainerArgs used when bringing up new autoforms.</param>
		///<param name="handle">The handle is used for data binding.</param>
		///<param name="umlClass">The <see cref="IModelElement"/> used to get the static information about the methods.</param>
		///<param name="table">The html table taht you want to add controls to.</param>
		///<param name="myMethodBindings">List of bindings between controls and methods.</param>
		///<param name="autoForm">The form responsible for presentation.</param>
		///<param name="controlPrefix">Prefix that will be prepended to the name of your control (used for recursive nestling of state machine regions).</param>
		///<param name="response">The HttpResponse from the asp page.</param>
		///<param name="request">The HttpRequest from the asp page.</param>
		///<param name="session">The HttpSessionState from the asp page.</param>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="hostControl"/> is null.</exception>
		public static void BuildControls(Control hostControl, EcoSpace ecoSpace,
			ElementHandle handle, IClass umlClass, Table table, IList myMethodBindings, AutoForm autoForm, string controlPrefix, HttpResponse response, HttpRequest request, HttpSessionState session)
		{
			if (hostControl == null)
				throw new ArgumentNullException("hostControl"); // do not localize
			if (umlClass == null)
				throw new ArgumentNullException("umlClass");

			foreach (IMethod method in umlClass.EcoClass.AllMethods)
			{
				if (method.EcoMethod.IsEcoCallable)
					BuildControlsForMethod(hostControl, ecoSpace, handle, method, table, myMethodBindings, autoForm, controlPrefix, response, request, session);
			}
		}

		///<summary>
		///Builds controls for a single <paramref name="method"/> on <paramref name="hostControl"/>.
		///</summary>
		///<param name="hostControl">The control used as container for created controls.</param>
		///<param name="ecoSpace">AutoContainerArgs used when bringing up new autoforms.</param>
		///<param name="handle">The handle is used for data binding.</param>
		///<param name="method">The <see cref="IMethod"/> used to get the static information about the parameters.</param>
		///<param name="table">The html table taht you want to add controls to.</param>
		///<param name="myMethodBindings">List of bindings between controls and methods.</param>
		///<param name="autoForm">The form responsible for presentation.</param>
		///<param name="controlPrefix">Prefix that will be prepended to the name of your control (used for recursive nestling of state machine regions).</param>
		///<param name="response">The HttpResponse from the asp page.</param>
		///<param name="request">The HttpRequest from the asp page.</param>
		///<param name="session">The HttpSessionState from the asp page.</param>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="hostControl"/> is null.</exception>
		public static void BuildControlsForMethod(Control hostControl, EcoSpace ecoSpace,
			ElementHandle handle, IMethod method, Table table, IList myMethodBindings, AutoForm autoForm, string controlPrefix, HttpResponse response, HttpRequest request, HttpSessionState session)
		{
			if (hostControl == null) 
				throw new ArgumentNullException("hostControl");
			if (method == null)
				throw new ArgumentNullException("method");
			TableRow tableRow = ControlBuilder.MakeNewTableRow(controlPrefix + "RowMeth" + method.Name, table); // do not localize

			OclVariables vars = BuildOclVariables(method, ecoSpace);

			BuildButton(hostControl, handle, method, vars, ecoSpace, tableRow, myMethodBindings, autoForm, controlPrefix, response, request, session);
		}

		private static OclVariables BuildOclVariables(IMethod method, EcoSpace ecoSpace)
		{
			OclVariables result = new OclVariables();
			foreach (IParameter p in method.Parameters)
			{
				VariableHandle vh = new VariableHandle();
				vh.StaticValueTypeName = p.Type.Name;
				vh.EcoSpace = ecoSpace;
				if (p.Type is ISimpleType)
				   vh.Element.AsObject = (p.Type as ISimpleType).EcoSimpleType.DefaultContent;

				// add a self-column
				OclColumn selfColumn = new OclColumn();
				selfColumn.Name = "self"; // do not localize
				selfColumn.Expression = "self"; // do not localize
				vh.Columns.Add(selfColumn);

				result.OclVariableCollection.Add(new OclVariable(p.Name, vh));
			}
			return result;
		}

		private static void BuildParameterHeader(IMethod method, Table table, string controlPrefix, HttpRequest request, HttpSessionState session)
		{
			TableRow row = ControlBuilder.MakeNewTableRow(controlPrefix + "RowParamHeader" + method.Name, table);// do not localize
			TableCell controlCell = ControlBuilder.MakeNewTableCell(controlPrefix + "CellControlParamHeader" + method.Name, row); // do not localize

			controlCell.ColumnSpan = 2;
			controlCell.Width = Unit.Percentage(100);
			controlCell.CssClass = EcoCssClassNames.EcoAutoFormParameterHeaderCell;

			WebUtils.AddHeaderTableToCell( controlCell, "Method" + method.Owner.Name +method.Name, WebStringRes.sParameters, request, session); // do not localize
		}

		private static void BuildParameterControls(IMethod method, OclVariables variables, Table table, IList parameterBindings, string controlPrefix)
		{
			foreach (OclVariable v in variables.OclVariableCollection)
			{
				TableRow row = ControlBuilder.MakeNewTableRow(controlPrefix + "RowParam" + method.Name + v.VariableName, table); // do not localize
				TableCell labelCell = ControlBuilder.MakeNewTableCell(controlPrefix + "CellLabelParam" + method.Name + v.VariableName, row); // do not localize
				TableCell controlCell = ControlBuilder.MakeNewTableCell(controlPrefix + "CellControlParam" + method.Name + v.VariableName, row); // do not localize

				Label label = new Label();
				label.Text = v.VariableName;
				labelCell.Controls.Add(label);

				string controlName = controlPrefix + "ecoParam" + method.Name + v.VariableName ; // do not localize
				Control ctrl = ControlBuilder.BuildControl( controlName, controlCell, v.ElementHandle, "self", v.ElementHandle.StaticUmlType, false, false); // do not localize
				parameterBindings.Add(new ParameterBinding(ctrl, v));
			}
		}

		private static string ParmList(IMethod method)
		{
			StringBuilder sb = new StringBuilder();
			sb.Append("("); // do not localize
			bool first = true;
			foreach (IParameter p in method.Parameters)
			{
				if (!first)
					sb.Append(','); // do not localize
				sb.Append(p.Name);
				first = false;
			}
			sb.Append(")"); // do not localize
			return sb.ToString();
		}

		private static void BuildButton(Control hostControl, ElementHandle handle, IMethod method, OclVariables variables, EcoSpace ecoSpace, TableRow tableRow, IList myMethodBindings, AutoForm autoForm, string controlPrefix, HttpResponse response, HttpRequest request, HttpSessionState session)
		{
			if (method == null) 
				throw new ArgumentNullException("method");
			Button button = new Button();
			button.CssClass = EcoCssClassNames.EcoAutoFormMethodButton;
			button.Text = method.Name;

			TableCell buttonCell = ControlBuilder.MakeNewTableCell(controlPrefix + "CellButtonMeth" + method.Name, tableRow); // do not localize
			TableCell controlsCell = ControlBuilder.MakeNewTableCell(controlPrefix + "CellControlsMeth" + method.Name, tableRow); // do not localize
			buttonCell.CssClass = EcoCssClassNames.EcoAutoFormLabelCell; // do not localize
			controlsCell.CssClass = EcoCssClassNames.EcoAutoFormControlCell; // do not localize
			controlsCell.Width = Unit.Percentage(100);

			IList parameterBindings = new ArrayList();

			if (method.Parameters.Count > 0)
			{
				Table paramTable = ControlBuilder.MakeNewTable(controlPrefix + "TableParamsMeth" + method.Name, controlsCell); // do not localize
				paramTable.Width = Unit.Percentage(100);

				BuildParameterHeader(method, paramTable, controlPrefix, request, session);

				if (!(WebUtils.IsCollapsedSection(session, "Method" + method.Owner.Name + method.Name))) // do not localize
					BuildParameterControls(method, variables, paramTable, parameterBindings, controlPrefix);
			}

			button.Enabled = IsMethodEnabled(handle, method, variables, ecoSpace);

			button.Click += new System.EventHandler(new ButtonExecuteMethod(handle, method, variables.OclVariableCollection, autoForm, parameterBindings, response, request).OnClick);
			buttonCell.Controls.Add(button);

			myMethodBindings.Add(new MethodButtonBinding(handle, method, variables, button, ecoSpace));
		}

		public static bool IsMethodEnabled(ElementHandle handle, IMethod method, OclVariables variables, EcoSpace ecoSpace)
		{
			if (method == null)
				throw new ArgumentNullException("method");
			if (ecoSpace == null)
				throw new ArgumentNullException("ecoSpace");
			ExpressionHandle canCallhandle = new ExpressionHandle();
			canCallhandle.RootHandle = handle;
			canCallhandle.Variables = variables;
			OclColumn selfColumn = new OclColumn();
			selfColumn.Name = "self"; // do not localize
			selfColumn.Expression = "self"; // do not localize
			canCallhandle.Columns.Add(selfColumn);

			bool enabled = false;

			if (method.Parameters.Count > 0)
				enabled = true;
			else
			{
				string parms = ParmList(method);
				canCallhandle.Expression = String.Format(CultureInfo.InvariantCulture, "{0}?{1}", method.Name, parms); // do not localize

				if (method.EcoMethod.Implementation != null || method.EcoMethod.IsTrigger || method.EcoMethod.MethodInfo != null)
				{
					IOclService oclService = (IOclService) ecoSpace.GetEcoService(typeof(IOclService));
					enabled = (bool) oclService.Evaluate(handle.Element, canCallhandle.Expression).AsObject;
				}
				else
				{
					// disable methods that have no implementation
					enabled = false;
				}
			}
			return enabled;
		}

	}
}
